const fs = require('fs');
const os = require('os');
const path = require('path');
const crypto = require('crypto');
const { spawn, exec } = require('child_process');
const { app } = require('electron');
const logger = require('./logger');

/**
 * Parse the decoder name from FFmpeg stderr output.
 * @param {string} stderr - FFmpeg stderr output
 * @returns {string} Decoder name or 'unknown'
 */
function parseDecoderFromOutput(stderr) {
  if (!stderr) return 'unknown';

  // IMPORTANT: Check for specific decoder strings FIRST before general patterns
  // The decoder in brackets like [h264_cuvid] or [libdav1d] is what we want

  // Priority 1: Look for explicit decoder patterns in brackets
  // FFmpeg outputs: "Decoder [decoder_name] (device/type): codec_name"
  let match = stderr.match(/Decoder\s*\[(\w+)\]/i);
  if (match) {
    return match[1]; // This will be like 'libdav1d', 'h264_cuvid', 'h264_d3d11va'
  }

  // Priority 2: Check for known GPU decoder substrings in the entire output
  const gpuDecoderPatterns = [
    'h264_cuvid', 'hevc_cuvid', 'av1_cuvid',    // NVIDIA
    'h264_nvenc', 'hevc_nvenc', 'av1_nvenc',
    'h264_d3d11va', 'hevc_d3d11va', 'av1_d3d11va', // Windows D3D11
    'h264_d3d12va', 'hevc_d3d12va', 'av1_d3d12va', // Windows D3D12
    'h264_dxva2', 'hevc_dxva2', 'av1_dxva2',   // Windows DXVA2
    'h264_vaapi', 'hevc_vaapi', 'av1_vaapi',   // Linux VAAPI
    'h264_qsv', 'hevc_qsv', 'av1_qsv',         // Intel QuickSync
    'h264_videotoolbox', 'hevc_videotoolbox',   // macOS
  ];

  for (const pattern of gpuDecoderPatterns) {
    if (stderr.includes(pattern)) {
      return pattern;
    }
  }

  // Priority 3: Check for libdav1d (AV1 software decoder)
  if (stderr.includes('libdav1d')) {
    return 'libdav1d';
  }

  // Priority 4: Check for other software decoders
  const swDecoders = ['libx264', 'libx265', 'libvpx', 'libaom'];
  for (const decoder of swDecoders) {
    if (stderr.includes(decoder)) {
      return decoder;
    }
  }

  // Priority 5: Fallback to codec name - but this is less accurate
  // Only use this if nothing else matched
  match = stderr.match(/Video:\s*(\S+)/i);
  if (match) {
    return match[1]; // This will be 'h264', 'av1', 'hevc' - the codec, not decoder
  }

  return 'unknown';
}

/**
 * Determine if the decoder is using GPU or CPU.
 * @param {string} decoder - Decoder name
 * @returns {string} 'GPU' or 'CPU'
 */
function getDecoderType(decoder) {
  if (!decoder || decoder === 'unknown') return 'unknown';
  if (/\b(cuda|d3d|dxva|vaapi|nvdec|videotoolbox|qsv)\b/i.test(decoder)) return 'GPU';
  if (decoder.includes('libdav1d')) return 'CPU';
  return 'CPU';
}

let ffmpegPath = null;

function initializeFfmpegPath() {
  try {
    const ffmpegStatic = require('ffmpeg-static');
    ffmpegPath = ffmpegStatic;

    if (ffmpegPath && ffmpegPath.includes('app.asar')) {
      const unpackedPath = ffmpegPath.replace('app.asar', 'app.asar.unpacked');
      if (fs.existsSync(unpackedPath)) {
        ffmpegPath = unpackedPath;
      }
    }

    logger.log('FFmpeg found at:', ffmpegPath);
  } catch (error) {
    logger.log('ffmpeg-static not available, will try to find ffmpeg manually');
  }
}

initializeFfmpegPath();

function setFfmpegPath(manualPath) {
  ffmpegPath = manualPath;
}

function getFfmpegPath() {
  return ffmpegPath;
}

function findFFmpeg() {
  const resourcesPath = process.resourcesPath || __dirname;
  const binaryName = process.platform === 'win32' ? 'ffmpeg.exe' : 'ffmpeg';
  const archFolders = [`${process.platform}-${process.arch}`];

  if (process.platform === 'darwin') {
    archFolders.push(process.arch === 'arm64' ? 'darwin-x64' : 'darwin-arm64');
  } else if (process.platform === 'win32') {
    archFolders.push(process.arch === 'ia32' ? 'win32-x64' : 'win32-ia32');
  }

  const candidatePaths = [];

  if (ffmpegPath) {
    candidatePaths.push(ffmpegPath);
    if (ffmpegPath.includes('app.asar')) {
      candidatePaths.push(ffmpegPath.replace('app.asar', 'app.asar.unpacked'));
    }
  }

  candidatePaths.push(
    path.join(resourcesPath, binaryName),
    path.join(resourcesPath, '..', binaryName),
    path.join(__dirname, '..', '..', binaryName)
  );

  archFolders.forEach((folder) => {
    candidatePaths.push(
      path.join(resourcesPath, 'app.asar.unpacked', 'node_modules', 'ffmpeg-static', binaryName),
      path.join(__dirname, '..', '..', 'app.asar.unpacked', 'node_modules', 'ffmpeg-static', binaryName),
      path.join(__dirname, '..', '..', 'node_modules', 'ffmpeg-static', binaryName),
      path.join(resourcesPath, 'app.asar.unpacked', 'node_modules', '@ffmpeg-installer', folder, binaryName),
      path.join(__dirname, '..', '..', 'app.asar.unpacked', 'node_modules', '@ffmpeg-installer', folder, binaryName),
      path.join(__dirname, '..', '..', 'node_modules', '@ffmpeg-installer', folder, binaryName)
    );
  });

  if (process.platform === 'darwin') {
    candidatePaths.push('/opt/homebrew/bin/ffmpeg', '/usr/local/bin/ffmpeg');
  } else if (process.platform === 'linux') {
    candidatePaths.push('/usr/bin/ffmpeg', '/usr/local/bin/ffmpeg');
  } else if (process.platform === 'win32') {
    const exeDir = path.dirname(process.execPath);
    candidatePaths.push(path.join(exeDir, 'ffmpeg.exe'));
  }

  for (const candidate of candidatePaths) {
    if (!candidate || !fs.existsSync(candidate)) {
      continue;
    }

    try {
      if (process.platform !== 'win32') {
        fs.accessSync(candidate, fs.constants.X_OK);
      }
      return candidate;
    } catch (accessError) {
      continue;
    }
  }

  return 'ffmpeg';
}

function executeFFmpeg(args) {
  return new Promise((resolve, reject) => {
    const ffmpegExecutable = findFFmpeg();
    const spawnOptions = process.platform === 'win32' ? { windowsHide: true } : {};
    const child = spawn(ffmpegExecutable, args, spawnOptions);

    let stderrOutput = '';

    if (child.stderr) {
      child.stderr.on('data', (data) => {
        stderrOutput += data.toString();
      });
    }

    child.on('error', (error) => {
      reject(error);
    });

    child.on('close', (code) => {
      if (code === 0) {
        resolve();
      } else {
        const errorMessage = stderrOutput ? stderrOutput.trim() : `FFmpeg exited with code ${code}`;
        reject(new Error(errorMessage));
      }
    });
  });
}

function formatPathForConcat(filePath) {
  if (process.platform === 'win32') {
    return filePath.replace(/\\/g, '\\\\').replace(/'/g, "\\'");
  }
  return filePath.replace(/'/g, "\\'");
}

/**
 * Get the video codec name for a video file using ffprobe.
 * @param {string} videoPath - Path to the video file
 * @returns {Promise<string|null>} Codec name (e.g., 'av1', 'h264', 'hevc') or null on failure
 */
async function getVideoCodec(videoPath) {
  try {
    const { findFFprobe } = require('./ffprobe');
    const ffprobe = findFFprobe();
    const { promisify } = require('util');
    const execAsync = promisify(exec);

    const escapedPath = JSON.stringify(videoPath);
    const command = `"${ffprobe}" -v quiet -print_format json -show_streams -select_streams v:0 ${escapedPath}`;

    const result = await execAsync(command);
    const info = JSON.parse(result.stdout);

    if (info.streams && info.streams.length > 0) {
      const codecName = info.streams[0].codec_name;
      logger.log(`[FFmpeg] Detected video codec: ${codecName} for ${path.basename(videoPath)}`);
      return codecName;
    }
    return null;
  } catch (error) {
    logger.warn(`[FFmpeg] Could not detect video codec for ${videoPath}:`, error.message);
    return null;
  }
}

/**
 * Check what hardware acceleration methods are available in FFmpeg.
 * @returns {Promise<string[]>} Array of available hwaccel methods
 */
async function getAvailableHwAccels() {
  try {
    const ffmpeg = findFFmpeg();
    const { promisify } = require('util');
    const execAsync = promisify(exec);

    const command = `"${ffmpeg}" -hide_banner -hwaccels`;
    const result = await execAsync(command);

    const lines = result.stdout.split('\n');
    const hwaccels = [];
    for (const line of lines) {
      const match = line.trim().match(/^(\w+)\s*$/);
      if (match && match[1] !== 'Hardware' && match[1] !== 'method' && match[1] !== '---') {
        hwaccels.push(match[1]);
      }
    }
    logger.log('[FFmpeg] Available hwaccel methods:', hwaccels.join(', '));
    return hwaccels;
  } catch (error) {
    logger.warn('[FFmpeg] Could not query hwaccels:', error.message);
    return [];
  }
}

/**
 * Get the best hardware decoder for a given codec.
 * @param {string} codec - Video codec (e.g., 'av1', 'h264', 'hevc')
 * @returns {Promise<{decoder: string, hwaccel: string}|null>} Decoder and hwaccel to use, or null if not available
 */
async function getHardwareDecoder(codec) {
  if (!codec) return null;

  const hwaccels = await getAvailableHwAccels();
  const codecLower = codec.toLowerCase();

  // Priority 1: NVIDIA CUVID (best performance)
  if (hwaccels.includes('cuda')) {
    const nvidiaDecoders = {
      'av1': 'av1_cuvid',
      'h264': 'h264_cuvid',
      'hevc': 'hevc_cuvid',
      'vp9': 'vp9_cuvid',
      'vp8': 'vp8_cuvid',
      'mpeg2': 'mpeg2_cuvid',
      'mpeg4': 'mpeg4_cuvid'
    };
    if (nvidiaDecoders[codecLower]) {
      return { decoder: nvidiaDecoders[codecLower], hwaccel: 'cuda' };
    }
  }

  // Priority 2: Intel QuickSync (QSV)
  if (hwaccels.includes('qsv')) {
    const qsvDecoders = {
      'av1': 'av1_qsv',
      'h264': 'h264_qsv',
      'hevc': 'hevc_qsv',
      'mpeg2': 'mpeg2_qsv'
    };
    if (qsvDecoders[codecLower]) {
      return { decoder: qsvDecoders[codecLower], hwaccel: 'qsv' };
    }
  }

  // Priority 3: D3D11VA (Windows)
  if (hwaccels.includes('d3d11va')) {
    const d3d11vaDecoders = {
      'h264': 'h264',
      'hevc': 'hevc',
      'mpeg2': 'mpeg2video',
      'mpeg4': 'mpeg4',
      'vp9': 'vp9',
      'wmv3': 'wmv3'
    };
    if (d3d11vaDecoders[codecLower]) {
      return { decoder: d3d11vaDecoders[codecLower], hwaccel: 'd3d11va' };
    }
  }

  // Priority 4: DXVA2 (Windows legacy)
  if (hwaccels.includes('dxva2')) {
    const dxva2Decoders = {
      'h264': 'h264',
      'hevc': 'hevc',
      'mpeg2': 'mpeg2video',
      'wmv3': 'wmv3',
      'vc1': 'vc1'
    };
    if (dxva2Decoders[codecLower]) {
      return { decoder: dxva2Decoders[codecLower], hwaccel: 'dxva2' };
    }
  }

  return null;
}

function generateThumbnail(videoPath, outputPath) {
  return new Promise(async (resolve, reject) => {
    const ffmpeg = findFFmpeg();
    const { spawn } = require('child_process');
    const spawnOptions = process.platform === 'win32' ? { windowsHide: true } : {};

    const codec = await getVideoCodec(videoPath);
    const hwaccels = await getAvailableHwAccels();
    const hwDecoder = await getHardwareDecoder(codec);

    // Build FFmpeg args with explicit hardware decoder if available
    const args = [];

    if (hwDecoder) {
      // Use explicit hardware acceleration
      args.push('-hwaccel', hwDecoder.hwaccel);
      logger.log(`[FFmpeg] Generating thumbnail for: ${path.basename(videoPath)}`);
      logger.log(`[FFmpeg]   Codec: ${codec || 'unknown'}, Using GPU decoder: ${hwDecoder.decoder} (${hwDecoder.hwaccel})`);
    } else {
      // Fallback to software decode
      args.push('-hwaccel', 'auto');
      logger.log(`[FFmpeg] Generating thumbnail for: ${path.basename(videoPath)}`);
      logger.log(`[FFmpeg]   Codec: ${codec || 'unknown'}, HW acceleration: auto (available: ${hwaccels.join(', ') || 'none'}) - using software decode`);
    }

    args.push('-ss', '00:00:01', '-i', videoPath);

    // Explicitly specify the decoder (needed for hardware decode)
    if (hwDecoder) {
      args.push('-c:v', hwDecoder.decoder);
    }

    args.push('-vframes', '1', '-vf', 'scale=320:-1', '-q:v', '2', outputPath, '-y');

    logger.log(`[FFmpeg]   Command: ${args.join(' ')}`);

    const child = spawn(ffmpeg, args, spawnOptions);
    let stderrOutput = '';

    child.stderr.on('data', (data) => {
      stderrOutput += data.toString();
    });

    child.on('error', (error) => {
      reject(error);
    });

    child.on('close', (code) => {
      // Use helper function to parse decoder from output
      const decoderUsed = parseDecoderFromOutput(stderrOutput);
      const hwStatus = getDecoderType(decoderUsed);
      const statusStr = decoderUsed === 'unknown' ? 'unknown' : (hwStatus === 'GPU' ? 'GPU' : 'CPU');

      if (decoderUsed === 'unknown') {
        // Log more of the stderr for debugging - capture more context
        const snippet = stderrOutput.substring(0, 1500).replace(/\n/g, ' | ');
        logger.log(`[FFmpeg]   Decoder: unknown - Full FFmpeg output: ${snippet}`);
      }
      logger.log(`[FFmpeg]   Decoder used: ${decoderUsed} (${statusStr})`);

      if (code === 0 && fs.existsSync(outputPath)) {
        logger.log(`[FFmpeg] Thumbnail generated successfully: ${outputPath}`);
        resolve(outputPath);
      } else {
        if (stderrOutput.includes('not found') || stderrOutput.includes('ENOENT')) {
          logger.error('FFmpeg not found. Please install FFmpeg and ensure it is in your system PATH, or place ffmpeg.exe in the application directory.');
        } else {
          logger.error('FFmpeg error:', stderrOutput);
        }
        reject(new Error(stderrOutput || `FFmpeg exited with code ${code}`));
      }
    });
  });
}

const AUDIO_SAMPLE_RATE = 16000;
const AUDIO_CHANNELS = 1;

/**
 * Extract audio from a video file to raw PCM (16-bit mono).
 * @param {string} videoPath - Path to the video file
 * @param {number} startTime - Start time in seconds
 * @param {number} endTime - End time in seconds
 * @returns {Promise<Buffer>} Raw PCM buffer (s16le)
 */
function extractAudioToPcm(videoPath, startTime, endTime) {
  return new Promise((resolve, reject) => {
    const duration = endTime - startTime;
    const ffmpegExecutable = findFFmpeg();
    const spawnOptions = process.platform === 'win32' ? { windowsHide: true } : {};
    const args = [
      '-y',
      '-hide_banner',
      '-loglevel', 'warning',
      '-ss', String(startTime),
      '-i', videoPath,
      '-t', String(duration),
      '-vn',
      '-acodec', 'pcm_s16le',
      '-ar', String(AUDIO_SAMPLE_RATE),
      '-ac', String(AUDIO_CHANNELS),
      '-f', 's16le',
      'pipe:1'
    ];

    logger.log(`[FFmpeg] Extracting audio ${startTime}s-${endTime}s (${duration}s) from ${videoPath}`);

    const child = spawn(ffmpegExecutable, args, { ...spawnOptions, stdio: ['ignore', 'pipe', 'pipe'] });
    const chunks = [];
    let stderrOutput = '';

    child.stdout.on('data', (chunk) => chunks.push(chunk));
    child.stderr.on('data', (data) => { stderrOutput += data.toString(); });

    child.on('error', reject);

    child.on('close', (code) => {
      if (code === 0) {
        const buf = Buffer.concat(chunks);
        logger.log(`[FFmpeg] Extracted ${buf.length} bytes (${buf.length / 2} samples)`);
        resolve(buf);
      } else {
        const errorMessage = stderrOutput ? stderrOutput.trim() : `FFmpeg exited with code ${code}`;
        reject(new Error(errorMessage));
      }
    });
  });
}

/**
 * Extract full video audio to a raw PCM file.
 * @param {string} videoPath - Path to the video file
 * @param {string} outputPath - Path for the output PCM file
 * @returns {Promise<void>}
 */
function extractFullAudioToFile(videoPath, outputPath) {
  const args = [
    '-y',
    '-hide_banner',
    '-loglevel', 'error',
    '-i', videoPath,
    '-vn',
    '-acodec', 'pcm_s16le',
    '-ar', String(AUDIO_SAMPLE_RATE),
    '-ac', String(AUDIO_CHANNELS),
    '-f', 's16le',
    outputPath
  ];
  return executeFFmpeg(args);
}

async function getThumbnail(videoPath) {
  if (!fs.existsSync(videoPath)) {
    logger.error('Video file does not exist:', videoPath);
    return null;
  }

  const thumbnailsDir = path.join(app.getPath('userData'), 'thumbnails');
  if (!fs.existsSync(thumbnailsDir)) {
    fs.mkdirSync(thumbnailsDir, { recursive: true });
  }

  const videoHash = crypto.createHash('md5').update(videoPath).digest('hex');
  const thumbnailPath = path.join(thumbnailsDir, `${videoHash}.jpg`);

  if (fs.existsSync(thumbnailPath)) {
    try {
      const videoStats = fs.statSync(videoPath);
      const thumbStats = fs.statSync(thumbnailPath);
      if (videoStats.mtime.getTime() <= thumbStats.mtime.getTime()) {
        return thumbnailPath;
      }
      fs.unlinkSync(thumbnailPath);
    } catch (error) {
      logger.error('Error checking thumbnail stats:', error);
      if (fs.existsSync(thumbnailPath)) {
        fs.unlinkSync(thumbnailPath);
      }
    }
  }

  try {
    await generateThumbnail(videoPath, thumbnailPath);
    if (fs.existsSync(thumbnailPath)) {
      return thumbnailPath;
    }
    logger.error('Thumbnail generation failed: file not created');
    return null;
  } catch (error) {
    logger.error('Error generating thumbnail for:', videoPath, error);
    return null;
  }
}

module.exports = {
  initializeFfmpegPath,
  setFfmpegPath,
  getFfmpegPath,
  findFFmpeg,
  executeFFmpeg,
  formatPathForConcat,
  generateThumbnail,
  getThumbnail,
  extractAudioToPcm,
  extractFullAudioToFile,
  getVideoCodec,
  getAvailableHwAccels,
  getHardwareDecoder,
  parseDecoderFromOutput,
  getDecoderType,
  AUDIO_SAMPLE_RATE,
  AUDIO_CHANNELS
};

